<?php
require_once __DIR__ . '/config.php';
require_once __DIR__ . '/functions.php';
require_once __DIR__ . '/db.php';

$upload_error = '';
$upload_success = '';

// Manejo de envío de CV
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['cv_submit'])) {
    $nombre     = trim($_POST['nombre'] ?? '');
    $email      = trim($_POST['email'] ?? '');
    $telefono   = trim($_POST['telefono'] ?? '');
    $vacante_id = !empty($_POST['vacante_id']) ? (int)$_POST['vacante_id'] : null;

    if (!$nombre || !$email || empty($_FILES['cv']['name'])) {
        $upload_error = "Todos los campos obligatorios deben completarse.";
    } else {
        $allowed = ['pdf','doc','docx'];
        $ext = strtolower(pathinfo($_FILES['cv']['name'], PATHINFO_EXTENSION));

        if (!in_array($ext, $allowed)) {
            $upload_error = "Formato no permitido. Solo PDF, DOC o DOCX.";
        } else {
            $uploadDir = __DIR__ . '/uploads/cv';
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0775, true);
            }

            $filename = time() . "_" . preg_replace("/[^a-zA-Z0-9_\.]/", "", $_FILES['cv']['name']);
            $target = $uploadDir . "/" . $filename;

            if (move_uploaded_file($_FILES['cv']['tmp_name'], $target)) {
                // Guardar CV
                $stmt = $pdo->prepare("INSERT INTO cvs (nombre, email, telefono, archivo, formato) VALUES (?,?,?,?,?)");
                $stmt->execute([
                    $nombre,
                    $email,
                    $telefono,
                    'uploads/cv/' . $filename,
                    $ext
                ]);

                $cv_id = $pdo->lastInsertId();

                // Postulación
                if ($vacante_id) {
                    $stmt2 = $pdo->prepare("INSERT INTO postulaciones (cv_id, vacante_id) VALUES (?,?)");
                    $stmt2->execute([$cv_id, $vacante_id]);
                }

                $_SESSION['cv_success'] = "Tu CV se ha enviado correctamente.";
                redirect('index.php?alert=cv_success');


            } else {
                $_SESSION['cv_error'] = "No se pudo subir el archivo. Inténtalo de nuevo.";
                redirect('index.php?alert=cv_error');


            }
        }
    }
}

// Cargar vacantes abiertas
$sqlVac = "SELECT v.*, e.nombre AS empresa
           FROM vacantes v
           INNER JOIN empresas e ON e.id = v.empresa_id
           WHERE v.estado = 'abierta' AND e.is_active = 1
           ORDER BY v.created_at DESC";
$vacantes = $pdo->query($sqlVac)->fetchAll(PDO::FETCH_ASSOC);

// Empresas con al menos una vacante abierta
$sqlEmp = "
    SELECT 
        e.*,

        -- Número de vacantes abiertas
        COUNT(CASE WHEN v.estado = 'abierta' THEN 1 END) AS vacantes_abiertas,

        -- Última descripción disponible (la más reciente)
        (
            SELECT v2.descripcion
            FROM vacantes v2
            WHERE v2.empresa_id = e.id
              AND v2.estado = 'abierta'
            ORDER BY v2.created_at DESC
            LIMIT 1
        ) AS descripcion_reciente,

        -- Última ubicación disponible
        (
            SELECT v3.ubicacion
            FROM vacantes v3
            WHERE v3.empresa_id = e.id
              AND v3.estado = 'abierta'
            ORDER BY v3.created_at DESC
            LIMIT 1
        ) AS ubicacion_principal,

        -- Último salario disponible
        (
            SELECT v4.salario
            FROM vacantes v4
            WHERE v4.empresa_id = e.id
              AND v4.estado = 'abierta'
            ORDER BY v4.created_at DESC
            LIMIT 1
        ) AS salario_principal

    FROM empresas e
    LEFT JOIN vacantes v 
        ON v.empresa_id = e.id
        AND v.estado = 'abierta'
    WHERE e.is_active = 1
    GROUP BY e.id
    HAVING vacantes_abiertas > 0
    ORDER BY e.nombre ASC
";

$empresas = $pdo->query($sqlEmp)->fetchAll(PDO::FETCH_ASSOC);


session_start();

$cv_success = $_SESSION['cv_success'] ?? '';
$cv_error   = $_SESSION['cv_error'] ?? '';

unset($_SESSION['cv_success'], $_SESSION['cv_error']);

?>

<!DOCTYPE html>
<html lang="es" dir="ltr">
<head>
  <!-- Meta SEO básico -->
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>SaaS RH Multiempresa | Vacantes & CVs</title>
  <meta name="description" content="Publica vacantes para múltiples empresas, recibe CVs y gestiona reclutamiento centralizado en un solo sistema." />
  <meta name="keywords" content="vacantes, reclutamiento, multiempresa, RH, subir CV" />

  <!-- Open Graph / Social -->
  <meta property="og:title" content="SaaS RH Multiempresa | Vacantes & CVs" />
  <meta property="og:description" content="Publica vacantes para varias empresas y recibe CVs en un solo panel." />
  <meta property="og:type" content="website" />
  <meta property="og:url" content="https://tu-dominio.com/" />
  <meta property="og:image" content="https://tu-dominio.com/og-image.jpg" />

  <!-- Twitter Card -->
  <meta name="twitter:card" content="summary_large_image" />
  <meta name="twitter:title" content="SaaS RH Multiempresa | Vacantes & CVs" />
  <meta name="twitter:description" content="Sistema de reclutamiento multiempresa." />
  <meta name="twitter:image" content="https://tu-dominio.com/og-image.jpg" />

  <!-- Favicon -->
  <link rel="icon" href="/favicon.ico" />
  
  <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>


  <!-- Estilos -->
  <style>
/* ===============================
   VARIABLES & BASE
=============================== */
:root {
  --bg: #0b1021;
  --panel: #121734;
  --text: #e6e9f5;
  --muted: #a9b1d6;
  --primary: #2f6cf6;
  --primary-700: #2557c8;
  --accent: #ff7a3d;
  --success: #19c37d;
  --warning: #f6c12f;
  --border: #243053;
}

* { box-sizing: border-box }
html, body { height: 100% }

body {
  margin: 0;
  padding-bottom: 90px;
  font-family: system-ui, -apple-system, Segoe UI, Roboto, Helvetica, Arial;
  background: linear-gradient(180deg, var(--bg), #0e1530);
  color: var(--text);
}

img { max-width: 100%; display: block }
a { color: inherit; text-decoration: none }

.container { max-width: 1120px; margin: 0 auto; padding: 0 20px }


/* ===============================
   NAV
=============================== */
.nav {
  position: sticky; top: 0;
  backdrop-filter: saturate(140%) blur(8px);
  background: rgba(11,16,33,0.7);
  border-bottom: 1px solid var(--border);
  z-index: 30;
}

.nav-inner {
  display: flex; justify-content: space-between;
  align-items: center; padding: 14px 0;
}

.logo { display: flex; align-items: center; gap: 10px; font-weight: 700; }
.logo-badge {
  width: 36px; height: 36px; border-radius: 10px;
  background: linear-gradient(135deg, var(--primary), var(--accent));
  display: grid; place-items: center; font-weight: 800;
}

.nav-links { display: none; gap: 18px }
@media (min-width: 860px) { .nav-links { display: flex } }

.cta, .cta-secondary {
  padding: 12px 18px; border-radius: 10px; font-weight: 700; cursor: pointer;
}

.cta {
  background: var(--primary); color: #fff;
  box-shadow: 0 8px 20px rgba(47,108,246,0.35);
}
.cta:hover { background: var(--primary-700) }

.cta-secondary {
  background: transparent; border: 1px solid var(--border);
}


/* ===============================
   HERO
=============================== */
.hero { padding: 64px 0 24px; text-align: center }
.pill {
  display: inline-flex; gap: 8px; align-items: center;
  padding: 8px 12px;
  border: 1px solid var(--border);
  border-radius: 999px;
  color: var(--muted);
  background: rgba(18,23,52,0.6);
}

.h1 { font-size: clamp(28px, 5vw, 46px); margin: 16px 0 12px }
.sub {
  font-size: 18px; color: var(--muted);
  max-width: 720px; margin: 0 auto 22px;
}

.hero-ctas {
  display: flex; gap: 14px; flex-wrap: wrap; justify-content: center;
}



/* ===============================
   HERO MEDIA (Laptop + iPhone)
=============================== */
.hero-media {
  margin-top: 100px;
  border: 1px solid var(--border);
  border-radius: 16px;
  padding: 20px;
  background: radial-gradient(1200px 500px at 50% -300px, rgba(47,108,246,0.15), transparent);
}

.hero-media-inner {
  display: flex;
  justify-content: center;
  align-items: center;
  gap: 40px;
  flex-wrap: nowrap;
}

@media (max-width: 860px) {
  .hero-media-inner {
    flex-direction: column;
    gap: 30px;
  }
}



/* ===============================
   LAPTOP
=============================== */
.laptop {
  width: 100%;
  max-width: 720px;
  background: #0f1735;
  border: 1px solid var(--border);
  border-radius: 14px;
  overflow: hidden;
  display: flex;
  flex-direction: column;
  box-shadow: 0 12px 34px rgba(0,0,0,0.45);
  padding: 0 !important;
}

.laptop-topbar {
  height: 38px;
  background: #0b0f1e;
  border-bottom: 1px solid #1a223d;
  display: flex; align-items: center; gap: 8px;
  padding: 0 14px;
}

.lap-dot { width: 12px; height: 12px; border-radius: 50% }
.lap-dot.red { background:#ff5f57 }
.lap-dot.yellow { background:#ffbd2e }
.lap-dot.green { background:#28c840 }

.laptop-screen {
  flex: 1;
  width: 100%;
  overflow: hidden;
  position: relative;
  padding: 0 !important;
  margin: 0 !important;
  line-height: 0;
}

.laptop-screen video {
  width: 100%;
  height: 100%;
  display: block;
  object-fit: cover;
  background: #000;
  margin: 0;
  padding: 0;
}



/* ===============================
   PHONE
=============================== */
.phone {
  width: 260px; background: #000;
  padding: 14px; border-radius: 34px;
  border: 2px solid #1f2a44;
  box-shadow: 0 15px 35px rgba(0,0,0,0.4);
}

.phone-notch {
  width: 120px; height: 26px; background: #000;
  border-bottom-left-radius: 16px; border-bottom-right-radius: 16px;
  margin: 0 auto;
}

.phone-screen {
  margin-top: 10px;
  background: #111;
  border-radius: 24px;
  overflow: hidden;
  width: 100%; height: 500px;
  position: relative;
}

.phone-screen video {
  width: 100%; height: 100%;
  object-fit: cover; display: block;
}

.phone-overlay {
  position: absolute;
  bottom: 0; left: 0; right: 0;
  padding: 12px 14px;
  background: linear-gradient(to top, rgba(0,0,0,0.85), transparent);
  color: #fff;
}



/* ===============================
   SECCIONES GENERALES
=============================== */
.section { padding: 54px 0 }
.section h2 { font-size: clamp(22px, 3.6vw, 32px); margin-bottom: 10px }
.section p.lead { color: var(--muted); max-width: 760px }

/* Ajuste general */
.section {
  padding: 80px 0;
  width: 100%;
  text-align: left;
}

.section h2 {
  font-size: clamp(26px, 3.6vw, 36px);
  font-weight: 800;
  margin-bottom: 16px;
  color: var(--text);
}

.section p.lead {
  font-size: 18px;
  color: var(--muted);
  margin-bottom: 26px;
  max-width: 760px;
}



/* ===========================
   BENEFICIOS
=========================== */
.features {
  display: grid;
  grid-template-columns: repeat(1, 1fr);
  gap: 20px;
}

@media (min-width: 860px) {
  .features {
    grid-template-columns: repeat(3, 1fr);
  }
}

.feature {
  background: #101733;
  border: 1px solid var(--border);
  border-radius: 14px;
  padding: 22px;
  transition: transform .2s ease, box-shadow .2s ease;
}

.feature:hover {
  transform: translateY(-4px);
  box-shadow: 0 10px 24px rgba(0,0,0,0.3);
}

.feature h3 {
  margin: 0 0 6px;
  font-size: 18px;
  font-weight: 700;
  color: #fff;
}

.feature p {
  color: var(--muted);
}



/* ===========================
   EMBUDO
=========================== */
.steps {
  display: grid;
  grid-template-columns: repeat(1, 1fr);
  gap: 20px;
  margin-top: 24px;
}

@media (min-width: 860px) {
  .steps {
    grid-template-columns: repeat(4, 1fr);
  }
}

.step {
  background: #101733;
  border: 1px solid var(--border);
  border-radius: 14px;
  padding: 22px;
}

.step .num {
  font-size: 22px;
  font-weight: 700;
  color: var(--accent);
  margin-bottom: 6px;
}



/* ===========================
   PLANES / PRICING
=========================== */
.pricing {
  display: grid;
  gap: 26px;
  grid-template-columns: repeat(1, 1fr);
  margin-top: 36px;
}

@media (min-width: 860px) {
  .pricing {
    grid-template-columns: repeat(3, 1fr);
  }
}

.price-card {
  background: #101733;
  border: 1px solid var(--border);
  border-radius: 14px;
  padding: 26px;
  display: flex;
  flex-direction: column;
  gap: 12px;
  transition: transform .2s ease, box-shadow .2s ease;
}

.price-card:hover {
  transform: translateY(-4px);
  box-shadow: 0 10px 24px rgba(0,0,0,0.3);
}

.price-card .caps {
  font-size: 13px;
  color: var(--muted);
}

.price {
  font-size: 32px;
  font-weight: 900;
}

.check {
  color: var(--success);
  font-weight: 700;
}



/* ===========================
   FAQ
=========================== */
.faq {
  display: grid;
  gap: 20px;
  margin-top: 30px;
  grid-template-columns: repeat(1, 1fr);
}

@media (min-width: 860px) {
  .faq {
    grid-template-columns: repeat(2, 1fr);
  }
}

.faq-item {
  background: #101733;
  border: 1px solid var(--border);
  border-radius: 14px;
  padding: 24px;
}

.faq-item h3 {
  margin: 0 0 8px;
  font-size: 18px;
}

.faq-item p {
  color: var(--muted);
}

/* FIX: Alinear pregunta “¿Listo para comenzar?” + botón */
.faq-call {
  display: flex;
  align-items: center;
  gap: 14px;
  margin-top: 10px;
}

.faq-call .cta {
  margin: 0 !important;
  box-shadow: 0 4px 14px rgba(0,0,0,0.35);
}



/* ===========================
   SECCIÓN EMPRESAS & VACANTES
=========================== */
.section-subtitle {
  font-size: 15px;
  color: var(--muted);
  margin-bottom: 10px;
}

.empresas-grid {
  display: grid;
  grid-template-columns: repeat(2, 1fr);
  gap: 16px;
}

@media (min-width: 860px) {
  .empresas-grid {
    grid-template-columns: repeat(4, 1fr);
  }
}

.empresa-card {
  background: #101733;
  border: 1px solid var(--border);
  border-radius: 14px;
  padding: 12px;
  text-align: center;
  cursor: pointer;
  transition: transform .2s ease, box-shadow .2s ease, border-color .2s ease;
}

.empresa-card:hover {
  transform: translateY(-4px);
  box-shadow: 0 10px 24px rgba(0,0,0,0.3);
  border-color: var(--primary);
}

.empresa-logo-wrap {
  padding: 4px 2px 8px;
}

.empresa-logo-wrap img {
  max-height: 60px;
  object-fit: contain;
  margin: 0 auto;
}

.empresa-name {
  font-size: 14px;
  font-weight: 600;
}

.tag-pill {
  display: inline-flex;
  align-items: center;
  gap: 6px;
  padding: 4px 10px;
  border-radius: 999px;
  border: 1px solid var(--border);
  font-size: 11px;
  color: var(--muted);
  background: #0f1329;
}

/* Vacantes */
.vacantes-header {
  display: flex;
  justify-content: space-between;
  align-items: baseline;
  gap: 10px;
  flex-wrap: wrap;
}

.vacantes-count {
  font-size: 13px;
  color: var(--muted);
}

/* Grid para desktop */
.vacantes-grid {
  display: none;
}

@media (min-width: 860px) {
  .vacantes-grid {
    display: grid;
    grid-template-columns: repeat(3, minmax(0,1fr));
    gap: 18px;
  }
}

/* Carrusel / scroll horizontal para móviles */
.vacantes-scroll {
  display: flex;
  gap: 14px;
  overflow-x: auto;
  scroll-snap-type: x mandatory;
  padding-bottom: 6px;
}

.vacantes-scroll::-webkit-scrollbar {
  height: 6px;
}
.vacantes-scroll::-webkit-scrollbar-thumb {
  background: #1f2937;
  border-radius: 999px;
}

.vacante-card {
  background: #101733;
  border: 1px solid var(--border);
  border-radius: 14px;
  padding: 16px;
  display: flex;
  flex-direction: column;
  gap: 6px;
  min-height: 180px;
}

.vacante-card-header {
  display: flex;
  justify-content: space-between;
  align-items: center;
  gap: 8px;
}

.badge-empresa {
  display: inline-flex;
  align-items: center;
  padding: 4px 10px;
  border-radius: 999px;
  border: 1px solid var(--border);
  background: #0b1024;
  font-size: 11px;
  color: var(--text);
}

.badge-estado {
  font-size: 11px;
  padding: 4px 8px;
  border-radius: 999px;
  background: rgba(25,195,125,0.1);
  color: #6ee7b7;
}

.vacante-title {
  font-size: 16px;
  font-weight: 700;
}

.vacante-meta {
  font-size: 12px;
  color: var(--muted);
}

.vacante-desc {
  font-size: 13px;
  color: var(--muted);
}

.btn-primary-ghost {
  margin-top: 6px;
  padding: 10px 14px;
  border-radius: 999px;
  border: 1px solid var(--primary);
  background: rgba(47,108,246,0.1);
  color: #fff;
  font-size: 14px;
  font-weight: 600;
  cursor: pointer;
  text-align: center;
}
.btn-primary-ghost:hover {
  background: rgba(47,108,246,0.25);
}

/* Para el scroll horizontal (móvil) */
.vacante-slide {
  min-width: 260px;
  scroll-snap-align: start;
}



/* ===============================
   FOOTER
=============================== */
.footer {
  padding: 36px 0;
  background: #090f1f;
  border-top: 1px solid var(--border);
  color: var(--muted);
}

.footer .container {
  display: flex; justify-content: space-between;
  flex-wrap: wrap; gap: 16px;
}

.footer a:hover { color: #fff }



/* ===============================
   WHATSAPP BUTTON
=============================== */
.whatsapp-float {
  position: fixed;
  bottom: 24px;
  right: 24px;
  width: 62px;
  height: 62px;
  background: linear-gradient(135deg, #25d366, #128c7e);
  border-radius: 50%;
  display: flex;
  align-items: center;
  justify-content: center;
  color: #fff;
  font-size: 32px;
  cursor: pointer;
  z-index: 9999;
  box-shadow: 0 8px 25px rgba(0,0,0,0.45);
  transition: transform .2s ease, box-shadow .2s ease;
  animation: wa-pop .35s ease-out;
}
.whatsapp-float:hover {
  transform: scale(1.08);
  box-shadow: 0 12px 28px rgba(0,0,0,0.55);
}
.wa-icon {
  width: 34px;
  height: 34px;
  fill: white;
}



/* ============================
   MODALES GENÉRICOS
============================ */
.modal-overlay {
  position: fixed;
  inset: 0;
  background: rgba(0,0,0,0.65);
  backdrop-filter: blur(6px);
  display: flex;
  align-items: center;
  justify-content: center;
  padding: 20px;
  z-index: 999999;
  opacity: 0;
  pointer-events: none;
  transition: opacity .25s ease;
}

.modal-overlay.active {
  opacity: 1;
  pointer-events: auto;
}

.modal {
  background: #101733;
  border: 1px solid var(--border);
  border-radius: 16px;
  padding: 24px;
  max-width: 560px;
  width: 100%;
  transform: scale(.85);
  transition: transform .25s ease;
  color: var(--text);
  position: relative;
}

.modal-overlay.active .modal {
  transform: scale(1);
}

.modal h2 {
  margin-top: 0;
  font-size: 22px;
  margin-bottom: 12px;
}

.modal p {
  color: var(--muted);
  line-height: 1.55;
  margin-bottom: 12px;
}

.modal-close {
  position: absolute;
  top: 12px;
  right: 18px;
  font-size: 22px;
  cursor: pointer;
  color: var(--muted);
  transition: 0.2s;
}

.modal-close:hover {
  color: #fff;
}

.modal input,
.modal textarea,
.modal select {
  width: 100%;
  padding: 12px;
  background: #0d1329;
  border: 1px solid var(--border);
  border-radius: 10px;
  color: var(--text);
  margin-bottom: 14px;
  font-size: 15px;
}

.modal button.cta {
  width: 100%;
}

/* Alertas en modal */
.alert {
  border-radius: 10px;
  padding: 10px 12px;
  font-size: 14px;
  margin-bottom: 10px;
}
.alert-success {
  background: rgba(25,195,125,0.12);
  border: 1px solid rgba(25,195,125,0.6);
  color: #bbf7d0;
}
.alert-danger {
  background: rgba(248,113,113,0.08);
  border: 1px solid rgba(248,113,113,0.6);
  color: #fecaca;
}

.carousel-wrapper {
  position: relative;
  width: 100%;
  margin-top: 20px;
}

.vacantes-carousel {
  display: flex;
  gap: 16px;
  overflow: hidden;
  scroll-behavior: smooth;
}

.vacante-slide {
  min-width: 280px;
  scroll-snap-align: center;
}

/* Botones */
.carousel-btn {
  position: absolute;
  top: 50%;
  transform: translateY(-50%);
  background: rgba(15, 20, 40, 0.8);
  border: 1px solid var(--border);
  color: white;
  width: 42px;
  height: 42px;
  border-radius: 50%;
  cursor: pointer;
  z-index: 10;
  font-size: 22px;
  display: flex;
  align-items: center;
  justify-content: center;
  transition: background .2s, transform .2s;
}

.carousel-btn:hover {
  background: rgba(47,108,246,0.5);
  transform: translateY(-50%) scale(1.1);
}

.carousel-btn.prev { left: -10px; }
.carousel-btn.next { right: -10px; }

@media (max-width: 600px) {
  .carousel-btn.prev { left: 0; }
  .carousel-btn.next { right: 0; }
}


  </style>
</head>

<body>

  <!-- NAV -->
  <header class="nav">
    <div class="container nav-inner">
      <div class="logo">
        <div class="logo-badge">RH</div>
        <img src="./Feedly-Col.png" style="height: 40px;" alt="Logo">
      </div>
      <nav class="nav-links">
        <a href="#beneficios">🧭 Beneficios</a>
        <a href="#embudo">📲 Cómo funciona</a>
        <a href="#pricing">💰 Precios</a>
        <a href="#faq">❓ FAQ</a>
        <a href="#empresas">🏢 Empresas</a>
        <a href="#vacantes">📄 Vacantes</a>
      </nav>
      <a class="cta"
         href="https://wa.me/528180659787?text=Hola,%20quiero%20información%20del%20SaaS%20RH%20Multiempresa"
         target="_blank">
        Hablar por WhatsApp
      </a>
    </div>
  </header>

  <!-- HERO -->
  <section class="hero">
    <div class="container">

      <div class="pill">
        <span>Reclutamiento Multiempresa</span><span>•</span>
        <span>Panel único de CVs</span><span>•</span>
        <span>Accesos por empresa</span>
      </div>

      <h1 class="h1">Centraliza vacantes de varias empresas en un solo sistema de RH</h1>
      <p class="sub">
        Publica vacantes, recibe CVs y controla qué ve cada empresa desde su propio panel.
        Ideal para agencias de reclutamiento o grupos empresariales.
      </p>

      <div class="hero-ctas">
        <button class="cta" type="button" onclick="openModal('modal-cv')">
          Subir mi CV
        </button>

        <button class="cta" type="button" onclick="scrollToId('vacantes')">
          Ver vacantes disponibles
        </button>
      </div>
      
      <!-- EMPRESAS CONTRATANDO -->
  <section id="empresas" class="section">
    <div class="container">
      <h2>Empresas contratando</h2>
      <!--p class="lead">Explora las empresas que ya publican vacantes en este sistema.</p-->

      <?php if (!$empresas): ?>
        <p class="section-subtitle">Aún no hay empresas con vacantes abiertas.</p>
      <?php else: ?>
        <p class="section-subtitle">
          Actualmente hay <strong><?php echo count($empresas); ?></strong> empresa(s) con vacantes abiertas.
        </p>

        <div class="empresas-grid">
  <?php foreach ($empresas as $e): ?>
    <div class="empresa-card"
         data-empresa-id="<?= (int)$e['id']; ?>"
         data-empresa-nombre="<?= h($e['nombre']); ?>">


<div class="empresa-logo-wrap" 
     style="width:100%;display:flex;justify-content:center;margin-bottom:10px;">

    <?php if (!empty($e['logo'])): ?>

        <?php 
            $logoPath = '/rh/admin/' . ltrim($e['logo'], '/');
        ?>

        <div style="
            width:100px;
            height:100px;
            border-radius:50%;
            overflow:hidden;
            background-image:url('<?= h($logoPath); ?>');
            background-size:cover;
            background-position:center;
            background-repeat:no-repeat;
            box-shadow:0 0 8px rgba(0,0,0,.15);
        "></div>

    <?php else: ?>

        <div style="
            width:100px;
            height:100px;
            border-radius:50%;
            background:#eaeaea;
            display:flex;
            align-items:center;
            justify-content:center;
            font-weight:bold;
            font-size:14px;
            text-align:center;
            padding:10px;
            box-shadow:0 0 8px rgba(0,0,0,.15);
        ">
            <?= h($e['nombre']); ?>
        </div>

    <?php endif; ?>
</div>




      <!-- TITULO -->
      <div class="empresa-name" style="margin-top:6px;font-size:15px;font-weight:700;">
        <?= h($e['nombre']); ?>
      </div>

      <!-- DESCRIPCIÓN CORTA >
      <div style="font-size:13px;color:var(--muted);margin-top:6px;">
        <?php 
          $desc = $e['descripcion'] ?? 'Empresa registrada en el sistema.';
          echo h(mb_strimwidth($desc, 0, 120, '…')); 
        ?>
      </div-->

      <!-- UBICACIÓN PRINCIPAL >
      <div style="font-size:12px;color:var(--muted);margin-top:6px;display:flex;align-items:center;gap:4px;">
        📍 <?= h($e['ubicacion_principal'] ?: 'Sin ubicación'); ?>
      </div-->

      <!-- SALARIO PROMEDIO >
      <div style="font-size:12px;color:var(--muted);margin-top:4px;display:flex;align-items:center;gap:4px;">
        💵 
        <?= $e['salario_promedio'] 
              ? '$'.number_format($e['salario_promedio'], 0)
              : 'Sin especificar'; ?>
      </div-->

      <!-- VACANTES -->
      <div class="tag-pill" style="margin-top:8px;">
        <span>Vacantes abiertas:</span>
        <strong><?= (int)$e['vacantes_abiertas']; ?></strong>
      </div>

    </div>
  <?php endforeach; ?>
</div>

      <?php endif; ?>
    </div>
  </section>

  <!-- VACANTES ABIERTAS -->
  <section id="vacantes" class="section">
    <div class="container">
      <div class="vacantes-header">
        <div>
          <h2>Vacantes abiertas</h2>
          <p class="section-subtitle">Postúlate a la vacante que mejor encaje contigo.</p>
        </div>
        <span class="vacantes-count">
          <?php echo count($vacantes); ?> vacante(s) disponible(s)
        </span>
      </div>

      <?php if (!$vacantes): ?>
        <p class="section-subtitle">Por el momento no hay vacantes activas.</p>
      <?php else: ?>
        <!-- Grid desktop >
        <div class="vacantes-grid">
          <?php foreach ($vacantes as $v): ?>
            <article class="vacante-card">
              <div class="vacante-card-header">
                <div>
                  <span class="badge-empresa"><?php echo h($v['empresa']); ?></span>
                </div>
                <span class="badge-estado">Abierta</span>
              </div>
              <div class="vacante-title"><?php echo h($v['titulo']); ?></div>
              <div class="vacante-meta">
                📍 <?php echo h($v['ubicacion'] ?: 'Ubicación no especificada'); ?> ·
                🏢 <?php echo h($v['modalidad'] ?: 'Modalidad no especificada'); ?>
              </div>
              <div class="vacante-desc">
                <?php
                  $desc = $v['descripcion'] ?? '';
                  $short = mb_strimwidth($desc, 0, 150, '…', 'UTF-8');
                  echo nl2br(h($short));
                ?>
              </div>
              <button class="btn-primary-ghost"
                      type="button"
                      data-vacante="<?php echo (int)$v['id']; ?>"
                      onclick="openCvForVacante(<?php echo (int)$v['id']; ?>)">
                Postularme
              </button>
            </article>
          <?php endforeach; ?>
        </div-->

        <!-- Scroll horizontal móvil -->
        <div class="carousel-wrapper">
  
  <!-- Botón anterior -->
  <button class="carousel-btn prev" onclick="moveCarousel(-1)">❮</button>

  <!-- Carrusel -->
  <div class="vacantes-carousel" id="vacantesCarousel">
    <?php foreach ($vacantes as $v): ?>
      <article class="vacante-card vacante-slide">
        <div class="vacante-card-header">
          <div><span class="badge-empresa"><?php echo h($v['empresa']); ?></span></div>
          <span class="badge-estado">Abierta</span>
        </div>

        <div class="vacante-title"><?php echo h($v['titulo']); ?></div>

        <div class="vacante-meta">
          📍 <?php echo h($v['ubicacion'] ?: 'Ubicación no especificada'); ?> ·
          🏢 <?php echo h($v['modalidad'] ?: 'Modalidad no especificada'); ?>
        </div>

        <div class="vacante-desc">
          <?php
            $desc = $v['descripcion'] ?? '';
            $short = mb_strimwidth($desc, 0, 150, '…', 'UTF-8');
            echo nl2br(h($short));
          ?>
        </div>

        <button class="btn-primary-ghost"
                type="button"
                data-vacante="<?php echo (int)$v['id']; ?>"
                onclick="openCvForVacante(<?php echo (int)$v['id']; ?>)">
          Postularme
        </button>
      </article>
    <?php endforeach; ?>
  </div>

  <!-- Botón siguiente -->
  <button class="carousel-btn next" onclick="moveCarousel(1)">❯</button>
</div>

      <?php endif; ?>
    </div>
  </section>

    
    </div>
  </section>

  <!-- BENEFICIOS -->
  <section id="beneficios" class="section">
    <div class="container">
      <h2>Beneficios clave para tu operación de RH</h2>
      <p class="lead">
        Gestiona vacantes para varias empresas, centraliza CVs y da acceso solo a quien le corresponde.
      </p>
      <div class="features">
        <div class="feature">
          <h3>🏢 Multiempresa real</h3>
          <p>
            Cada empresa solo ve sus vacantes y sus postulaciones.
            Ideal para grupos empresariales o agencias.
          </p>
        </div>
        <div class="feature">
          <h3>📥 CVs centralizados</h3>
          <p>
            Todos los CVs llegan a un solo sistema, organizados por vacante y empresa.
          </p>
        </div>
        <div class="feature">
          <h3>👤 Panel por empresa</h3>
          <p>
            Usuarios administradores por empresa con acceso limitado
            únicamente a sus datos.
          </p>
        </div>
        <div class="feature">
          <h3>⚙️ Configuración sencilla</h3>
          <p>
            Da de alta nuevas empresas y vacantes en minutos,
            sin desarrollos extra.
          </p>
        </div>
        <div class="feature">
          <h3>🚀 Escalable</h3>
          <p>
            Agrega nuevas marcas o clientes sin cambiar de sistema.
          </p>
        </div>
        <div class="feature">
          <h3>🔐 Accesos controlados</h3>
          <p>
            Rol de superadmin y administradores por empresa,
            con visibilidad segmentada.
          </p>
        </div>
      </div>
    </div>
  </section>

  <!-- EMBUDO -->
  <section id="embudo" class="section">
    <div class="container">
      <h2>Cómo funciona el flujo de reclutamiento</h2>
      <p class="lead">Desde la publicación de vacantes hasta la recepción de CVs en un solo panel.</p>

      <div class="steps">
        <div class="step">
          <div class="num">01</div>
          <h3>Publica vacantes</h3>
          <p>Das de alta vacantes por empresa con ubicación, modalidad, tipo de puesto y salario.</p>
        </div>
        <div class="step">
          <div class="num">02</div>
          <h3>Los candidatos aplican</h3>
          <p>Los candidatos ven las vacantes por empresa y se postulan subiendo su CV.</p>
        </div>
        <div class="step">
          <div class="num">03</div>
          <h3>Centralizas CVs</h3>
          <p>Todos los CVs llegan a un solo sistema, vinculados a la vacante y empresa correcta.</p>
        </div>
        <div class="step">
          <div class="num">04</div>
          <h3>Accesos por empresa</h3>
          <p>Cada empresa entra a su panel y ve únicamente sus vacantes y postulaciones.</p>
        </div>
      </div>
    </div>
  </section>

  <!-- PRECIOS -->
  <section id="pricing" class="section">
    <div class="container">
      <h2>Planes simples para empezar</h2>
      <p class="lead">Sin demos eternas. Empieza a publicar vacantes hoy mismo.</p>
      <div class="pricing">
        <div class="price-card" style="border-color: var(--accent)">
          <div class="caps">Starter</div>
          <div class="price">$30 USD / mes</div>
          <div><span class="check">✓</span> Multiempresa limitado</div>
          <div><span class="check">✓</span> Panel de administración</div>
          <div><span class="check">✓</span> Soporte básico</div>
        </div>
        <div class="price-card">
          <div class="caps">Pro - Pago único</div>
          <div class="price">$350 USD</div>
          <div><span class="check">✓</span> Instancia en tu propio hosting</div>
          <div><span class="check">✓</span> 1 mes de soporte</div>
          <div><span class="check">✓</span> Personalizaciones básicas</div>
        </div>
        <div class="price-card">
          <div class="caps">Enterprise</div>
          <div class="price">A medida</div>
          <div><span class="check">✓</span> Funcionalidades nuevas</div>
          <div><span class="check">✓</span> Integraciones extra</div>
          <div><span class="check">✓</span> Marca blanca / reseller</div>
        </div>
      </div>
    </div>
  </section>

  <!-- FAQ -->
  <section id="faq" class="section">
    <div class="container">
      <h2>Preguntas frecuentes</h2>
      <div class="faq">
        <div class="faq-item">
          <h3>🟢 ¿Cuánto tarda la instalación?</h3>
          <p>En menos de 2 horas puedes tener tu sistema listo si usamos nuestro hosting. En tu hosting puede variar según acceso.</p>
        </div>
        <div class="faq-item">
          <h3>🟢 ¿Puedo manejar varias empresas?</h3>
          <p>Sí, el sistema está pensado para multiempresa. Cada empresa tiene su propio login y solo ve sus datos.</p>
        </div>
        <div class="faq-item">
          <h3>🟢 ¿Quién ve los CVs?</h3>
          <p>El superadmin ve todo y cada empresa ve únicamente los CVs asociados a sus vacantes.</p>
        </div>
        <div class="faq-item">
          <h3>🟢 ¿Puedo crecer en número de empresas?</h3>
          <p>Sí, puedes ir agregando más empresas desde el panel (módulo de superadmin) y el sistema escala sin problemas.</p>
        </div>
      </div>

      <div class="faq-call">
        <strong>¿Listo para comenzar?</strong>
        <a class="cta"
           href="https://wa.me/528180659787?text=Hola,%20quiero%20instalar%20el%20SaaS%20RH%20Multiempresa"
           target="_blank">
          Hablar por WhatsApp
        </a>
      </div>
    </div>
  </section>

  

  <!-- FOOTER -->
  <footer class="footer">
    <div class="container">
      <div>© <span id="year"></span> SaaS RH Multiempresa</div>
      <div style="display:flex;gap:16px;flex-wrap:wrap;">
        <a href="#" onclick="openModal('modal-terminos')">Términos</a>
        <a href="#" onclick="openModal('modal-privacidad')">Privacidad</a>
        <a href="#" onclick="openModal('modal-contacto')">Contacto</a>
      </div>
    </div>
  </footer>

  <!-- BOTÓN WHATSAPP -->
  <a href="https://wa.me/528180659787?text=Hola,%20quiero%20información%20del%20SaaS%20RH%20Multiempresa"
     class="whatsapp-float" target="_blank" aria-label="Abrir chat de WhatsApp">
    <svg viewBox="0 0 32 32" class="wa-icon">
      <path d="M16.001 3.2c-7.061 0-12.8 5.739-12.8 12.8 0 2.259.589 4.466 1.709 6.41L3.2 28.8l6.586-1.689c1.874.996 3.987 1.523 6.215 1.523h.001c7.061 0 12.8-5.739 12.8-12.8s-5.739-12.8-12.801-12.8zm0 23.467h-.001c-1.992 0-3.938-.53-5.644-1.533l-.403-.238-3.909 1.002 1.045-3.808-.263-.392c-1.067-1.593-1.63-3.447-1.63-5.397 0-5.48 4.458-9.934 9.934-9.934 2.652 0 5.145 1.034 7.02 2.91 1.875 1.875 2.912 4.368 2.912 7.02 0 5.48-4.458 9.934-9.934 9.934zm5.488-7.458c-.3-.149-1.767-.871-2.04-.971-.273-.1-.472-.149-.671.15-.2.298-.771.97-.945 1.169-.174.2-.348.224-.647.075-.299-.15-1.264-.466-2.406-1.486-.889-.793-1.489-1.771-1.663-2.07-.174-.298-.019-.459.131-.608.135-.134.299-.348.448-.522.149-.174.199-.299.299-.498.099-.199.05-.373-.025-.522-.075-.149-.671-1.612-.919-2.207-.242-.579-.487-.5-.671-.51-.173-.008-.372-.01-.571-.01-.199 0-.522.075-.795.373-.273.298-1.045 1.022-1.045 2.493 0 1.471 1.07 2.891 1.219 3.09.149.199 2.108 3.221 5.107 4.515.714.308 1.27.492 1.705.63.716.227 1.368.195 1.883.118.575-.085 1.767-.722 2.017-1.419.249-.696.249-1.293.174-1.419-.074-.124-.273-.199-.572-.348z"/>
    </svg>
  </a>

  <!-- ============================
     MODAL: TÉRMINOS
============================ -->
  <div class="modal-overlay" id="modal-terminos" onclick="overlayClose(event, 'modal-terminos')">
    <div class="modal">
      <div class="modal-close" onclick="closeModal('modal-terminos')">✕</div>
      <h2>Términos del Servicio</h2>
      <p>Al usar este sistema aceptas que es software listo para producción. Está prohibida la reventa del código sin autorización.</p>
      <p>El usuario es responsable de la información, configuraciones y uso final del sistema.</p>
    </div>
  </div>

  <!-- ============================
     MODAL: PRIVACIDAD
============================ -->
  <div class="modal-overlay" id="modal-privacidad" onclick="overlayClose(event, 'modal-privacidad')">
    <div class="modal">
      <div class="modal-close" onclick="closeModal('modal-privacidad')">✕</div>
      <h2>Política de Privacidad</h2>
      <p>La información de candidatos y empresas se utiliza únicamente para el funcionamiento del sistema de reclutamiento.</p>
      <p>No se comparte con terceros sin consentimiento.</p>
    </div>
  </div>

  <!-- ============================
     MODAL: CONTACTO
============================ -->
  <div class="modal-overlay" id="modal-contacto" onclick="overlayClose(event, 'modal-contacto')">
    <div class="modal">
      <div class="modal-close" onclick="closeModal('modal-contacto')">✕</div>
      <h2>Contáctanos</h2>
      <p>Déjanos tus datos y te respondemos en minutos:</p>

      <input type="text" id="c-nombre" placeholder="Tu nombre" required>
      <input type="email" id="c-email" placeholder="Tu correo" required>
      <textarea id="c-msg" rows="4" placeholder="Tu mensaje"></textarea>

      <button class="cta" onclick="sendContact()">Enviar mensaje</button>

      <p id="contactSuccess" style="display:none;margin-top:10px;color:#19c37d;font-weight:600;">
        ✔ Tu mensaje fue enviado. Te contactaremos pronto.
      </p>
    </div>
  </div>

  <!-- ============================
     MODAL: VACANTES POR EMPRESA
============================ -->
  <div class="modal-overlay" id="modal-empresa" onclick="overlayClose(event, 'modal-empresa')">
    <div class="modal">
      <div class="modal-close" onclick="closeModal('modal-empresa')">✕</div>
      <h2>Vacantes en <span id="empresaModalNombre"></span></h2>
      <div id="empresaModalVacantes" style="margin-top:10px;"></div>
      <p id="empresaModalEmpty" class="section-subtitle" style="margin-top:12px;display:none;">
        No hay vacantes abiertas en esta empresa.
      </p>
    </div>
  </div>

  <!-- ============================
     MODAL: SUBIR CV / POSTULACIÓN
============================ -->
  <div class="modal-overlay" id="modal-cv" onclick="overlayClose(event, 'modal-cv')">
    <form class="modal" method="post" enctype="multipart/form-data" onsubmit="return onCvSubmit()">
      <div class="modal-close" onclick="closeModal('modal-cv')">✕</div>
      <h2>Postularme / Subir CV</h2>

      <?php if ($upload_error): ?>
        <div class="alert alert-danger"><?php echo h($upload_error); ?></div>
      <?php elseif ($upload_success): ?>
        <div class="alert alert-success"><?php echo h($upload_success); ?></div>
      <?php endif; ?>

      <input type="hidden" name="cv_submit" value="1">
      <input type="hidden" name="vacante_id" id="vacante_id_modal" value="">

      <label>Nombre completo</label>
      <input type="text" name="nombre" required>

      <label>Correo</label>
      <input type="email" name="email" required>

      <label>Teléfono</label>
      <input type="text" name="telefono">

      <label>Vacante (opcional)</label>
      <select name="vacante_select" onchange="syncVacante(this.value)">
        <option value="">-- Selecciona --</option>
        <?php foreach ($vacantes as $v): ?>
          <option value="<?php echo (int)$v['id']; ?>">
            <?php echo h($v['titulo'] . ' · ' . $v['empresa']); ?>
          </option>
        <?php endforeach; ?>
      </select>

      <div class="tag-pill" style="justify-content:space-between;margin-bottom:10px;">
        <span><strong>Empresa:</strong> <span id="cv_empresa_label">Sin empresa seleccionada</span></span>
      </div>
      <div class="tag-pill" style="justify-content:space-between;margin-bottom:14px;">
        <span><strong>Vacante:</strong> <span id="cv_vacante_label">Sin vacante seleccionada</span></span>
      </div>

      <label>Archivo CV (PDF, DOC, DOCX)</label>
      <input type="file" name="cv" required>

      <button class="cta" type="submit">Enviar CV</button>
    </form>
  </div>
  
  
  

  <!-- SCRIPTS -->
  
  <script>
  const carousel = document.getElementById("vacantesCarousel");
  let autoScroll;

  function moveCarousel(direction) {
    const slideWidth = carousel.children[0].offsetWidth + 16; // ancho + gap
    
    // Movimiento normal
    carousel.scrollLeft += slideWidth * direction;

    // Si llegó al final → Reinicia al inicio
    const maxScroll = carousel.scrollWidth - carousel.clientWidth;

    if (carousel.scrollLeft >= maxScroll - 5) {
      setTimeout(() => { 
        carousel.scrollLeft = 0;
      }, 300); // pequeño delay suave
    }

    // Si está al inicio y retrocede → Salta al final
    if (carousel.scrollLeft <= 0 && direction === -1) {
      carousel.scrollLeft = maxScroll;
    }
  }

  function startAutoScroll() {
    autoScroll = setInterval(() => moveCarousel(1), 3000); // cada 3s
  }

  function stopAutoScroll() {
    clearInterval(autoScroll);
  }

  // Auto-Scroll
  startAutoScroll();

  // Pausa al pasar el mouse
  carousel.addEventListener("mouseenter", stopAutoScroll);
  carousel.addEventListener("mouseleave", startAutoScroll);
</script>

  
  <script>
    function scrollToId(id) {
      const el = document.getElementById(id);
      if (el) el.scrollIntoView({ behavior: 'smooth' });
    }

    document.getElementById('year').textContent = new Date().getFullYear();

    // Utilidad para modales
    function openModal(id) {
      const el = document.getElementById(id);
      if (el) el.classList.add('active');
    }
    function closeModal(id) {
      const el = document.getElementById(id);
      if (el) el.classList.remove('active');
    }
    function overlayClose(ev, id) {
      if (ev.target.id === id) {
        closeModal(id);
      }
    }

    // Contacto dummy
    function sendContact() {
      const success = document.getElementById('contactSuccess');
      if (success) success.style.display = 'block';
      setTimeout(() => {
        if (success) success.style.display = 'none';
      }, 3000);
    }

    // Datos de vacantes en JS
    const VACANTES_DATA = <?php echo json_encode($vacantes, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES); ?>;
    const VACANTES_BY_ID = {};
    VACANTES_DATA.forEach(v => { VACANTES_BY_ID[v.id] = v; });

    function setVacanteInfo(id) {
      const empresaLabel = document.getElementById('cv_empresa_label');
      const vacanteLabel = document.getElementById('cv_vacante_label');
      const v = VACANTES_BY_ID[id];
      if (v) {
        if (empresaLabel) empresaLabel.textContent = v.empresa || 'Empresa sin nombre';
        if (vacanteLabel) vacanteLabel.textContent = v.titulo || 'Vacante sin título';
      } else {
        if (empresaLabel) empresaLabel.textContent = 'Sin empresa seleccionada';
        if (vacanteLabel) vacanteLabel.textContent = 'Sin vacante seleccionada';
      }
    }

    function syncVacante(id) {
      const hidden = document.getElementById('vacante_id_modal');
      if (hidden) hidden.value = id || '';
      setVacanteInfo(id || null);
    }

    function openCvForVacante(id) {
      const hidden = document.getElementById('vacante_id_modal');
      const selectVac = document.querySelector('select[name="vacante_select"]');
      if (hidden) hidden.value = id;
      if (selectVac) selectVac.value = id;
      setVacanteInfo(id);
      openModal('modal-cv');
    }

    function escapeHtml(str) {
      if (!str) return '';
      return str.replace(/[&<>"']/g, function(m) {
        return ({
          '&':'&amp;',
          '<':'&lt;',
          '>':'&gt;',
          '"':'&quot;',
          "'":'&#39;'
        })[m];
      });
    }

    // Tarjetas de empresas -> abrir modal con sus vacantes
    const empresaCards = document.querySelectorAll('.empresa-card');
    empresaCards.forEach(card => {
      card.addEventListener('click', () => {
        const empresaId = card.getAttribute('data-empresa-id');
        const empresaNombre = card.getAttribute('data-empresa-nombre') || '';
        const titleSpan = document.getElementById('empresaModalNombre');
        const listContainer = document.getElementById('empresaModalVacantes');
        const emptyMsg = document.getElementById('empresaModalEmpty');

        if (titleSpan) titleSpan.textContent = empresaNombre;
        if (listContainer) listContainer.innerHTML = '';

        const vacs = VACANTES_DATA.filter(v => String(v.empresa_id) === String(empresaId));

        if (!vacs.length) {
          if (emptyMsg) emptyMsg.style.display = 'block';
        } else {
          if (emptyMsg) emptyMsg.style.display = 'none';
          vacs.forEach(v => {
            const wrapper = document.createElement('div');
            wrapper.style.border = '1px solid var(--border)';
            wrapper.style.borderRadius = '12px';
            wrapper.style.padding = '12px 14px';
            wrapper.style.marginBottom = '10px';
            wrapper.style.background = '#0f1329';

            const desc = v.descripcion || '';
            const shortDesc = desc.length > 180 ? desc.substring(0,180) + '…' : desc;

            wrapper.innerHTML = `
              <div style="display:flex;justify-content:space-between;align-items:center;margin-bottom:4px;">
                <strong>${escapeHtml(v.titulo)}</strong>
                <span class="badge-estado">Abierta</span>
              </div>
              <div class="vacante-meta" style="margin-bottom:4px;">
                📍 ${escapeHtml(v.ubicacion || 'Ubicación no especificada')} ·
                🏢 ${escapeHtml(v.modalidad || 'Modalidad no especificada')}
              </div>
              <div class="vacante-desc" style="margin-bottom:8px;">${escapeHtml(shortDesc)}</div>
              <button type="button" class="btn-primary-ghost" data-vacante="${v.id}">
                Postularme a esta vacante
              </button>
            `;

            const btn = wrapper.querySelector('button');
            btn.addEventListener('click', () => {
              openCvForVacante(v.id);
              closeModal('modal-empresa');
            });

            listContainer.appendChild(wrapper);
          });
        }

        openModal('modal-empresa');
      });
    });

    // Para evitar doble submit si ya se envió bien (simple)
    function onCvSubmit() {
      // puedes agregar validaciones extra aquí si quieres
      return true;
    }
    
    
  </script>
  
  
  <?php if ($cv_success): ?>
<script>
Swal.fire({
    icon: 'success',
    title: '¡CV enviado!',
    text: '<?= $cv_success ?>',
    confirmButtonColor: '#2f6cf6'
});
</script>
<?php endif; ?>

<?php if ($cv_error): ?>
<script>
Swal.fire({
    icon: 'error',
    title: 'Error',
    text: '<?= $cv_error ?>',
    confirmButtonColor: '#d33'
});
</script>
<?php endif; ?>


</body>
</html>
