<?php
// Instalador sencillo para SaaS RH Multiempresa

if (file_exists(__DIR__ . '/config.php') && strpos(file_get_contents(__DIR__ . '/config.php'), 'DB_HOST') !== false) {
    die('El sistema ya está instalado. Si quieres reinstalar, borra config.php manualmente.');
}

$error = '';
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $db_host   = trim($_POST['db_host'] ?? '');
    $db_name   = trim($_POST['db_name'] ?? '');
    $db_user   = trim($_POST['db_user'] ?? '');
    $db_pass   = trim($_POST['db_pass'] ?? '');
    $base_url  = rtrim(trim($_POST['base_url'] ?? ''), '/');
    $sa_name   = trim($_POST['sa_name'] ?? '');
    $sa_email  = trim($_POST['sa_email'] ?? '');
    $sa_pass   = trim($_POST['sa_pass'] ?? '');

    if (!$db_host || !$db_name || !$db_user || !$base_url || !$sa_name || !$sa_email || !$sa_pass) {
        $error = 'Todos los campos son obligatorios.';
    } else {
        try {
            $dsn = 'mysql:host=' . $db_host . ';dbname=' . $db_name . ';charset=utf8mb4';
            $pdo = new PDO($dsn, $db_user, $db_pass, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            ]);

            // Limpiar tablas previas
            $pdo->exec('SET FOREIGN_KEY_CHECKS=0');
            $pdo->exec('DROP TABLE IF EXISTS postulaciones');
            $pdo->exec('DROP TABLE IF EXISTS cvs');
            $pdo->exec('DROP TABLE IF EXISTS vacantes');
            $pdo->exec('DROP TABLE IF EXISTS users');
            $pdo->exec('DROP TABLE IF EXISTS empresas');
            $pdo->exec('SET FOREIGN_KEY_CHECKS=1');

            // Crear tablas
            $sqls = [];

            $sqls[] = "CREATE TABLE empresas (
                id INT(11) NOT NULL AUTO_INCREMENT,
                nombre VARCHAR(255) NOT NULL,
                slug VARCHAR(100) DEFAULT NULL,
                logo VARCHAR(255) DEFAULT NULL,
                color_primario VARCHAR(20) DEFAULT NULL,
                is_active TINYINT(1) NOT NULL DEFAULT 1,
                created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (id)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";

            $sqls[] = "CREATE TABLE users (
                id INT(11) NOT NULL AUTO_INCREMENT,
                empresa_id INT(11) DEFAULT NULL,
                nombre VARCHAR(255) NOT NULL,
                email VARCHAR(190) NOT NULL,
                password_hash VARCHAR(255) NOT NULL,
                role TINYINT(1) NOT NULL DEFAULT 2, -- 1 superadmin, 2 admin empresa
                is_active TINYINT(1) NOT NULL DEFAULT 1,
                created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (id),
                UNIQUE KEY uq_email (email),
                KEY idx_empresa (empresa_id),
                CONSTRAINT fk_user_empresa FOREIGN KEY (empresa_id) REFERENCES empresas(id)
                    ON DELETE SET NULL ON UPDATE CASCADE
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";

            $sqls[] = "CREATE TABLE vacantes (
                id INT(11) NOT NULL AUTO_INCREMENT,
                empresa_id INT(11) NOT NULL,
                titulo VARCHAR(255) NOT NULL,
                descripcion TEXT,
                ubicacion VARCHAR(255),
                modalidad VARCHAR(100),
                tipo VARCHAR(100),
                salario VARCHAR(100),
                estado ENUM('abierta','cerrada') NOT NULL DEFAULT 'abierta',
                created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (id),
                KEY idx_empresa (empresa_id),
                CONSTRAINT fk_vacante_empresa FOREIGN KEY (empresa_id) REFERENCES empresas(id)
                    ON DELETE CASCADE ON UPDATE CASCADE
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";

            $sqls[] = "CREATE TABLE cvs (
                id INT(11) NOT NULL AUTO_INCREMENT,
                nombre VARCHAR(255) NOT NULL,
                email VARCHAR(190) NOT NULL,
                telefono VARCHAR(50),
                archivo VARCHAR(255) NOT NULL,
                formato VARCHAR(10) NOT NULL,
                created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (id)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";

            $sqls[] = "CREATE TABLE postulaciones (
                id INT(11) NOT NULL AUTO_INCREMENT,
                cv_id INT(11) NOT NULL,
                vacante_id INT(11) NOT NULL,
                created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (id),
                KEY idx_cv (cv_id),
                KEY idx_vacante (vacante_id),
                CONSTRAINT fk_post_cv FOREIGN KEY (cv_id) REFERENCES cvs(id)
                    ON DELETE CASCADE ON UPDATE CASCADE,
                CONSTRAINT fk_post_vacante FOREIGN KEY (vacante_id) REFERENCES vacantes(id)
                    ON DELETE CASCADE ON UPDATE CASCADE
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";

            foreach ($sqls as $sql) {
                $pdo->exec($sql);
            }

            // Crear superadmin
            $hash = password_hash($sa_pass, PASSWORD_BCRYPT);
            $stmt = $pdo->prepare('INSERT INTO users (empresa_id, nombre, email, password_hash, role, is_active)
                                   VALUES (NULL, ?, ?, ?, 1, 1)');
            $stmt->execute([$sa_name, $sa_email, $hash]);

            // Crear config.php
            $config = "<?php\n";
            $config .= "define('DB_HOST', '" . addslashes($db_host) . "');\n";
            $config .= "define('DB_NAME', '" . addslashes($db_name) . "');\n";
            $config .= "define('DB_USER', '" . addslashes($db_user) . "');\n";
            $config .= "define('DB_PASS', '" . addslashes($db_pass) . "');\n";
            $config .= "define('BASE_URL', '" . addslashes($base_url) . "');\n";
            $config .= "define('APP_NAME', 'SaaS RH Multiempresa');\n";

            file_put_contents(__DIR__ . '/config.php', $config);

            $success = 'Instalación correcta. Ya puedes iniciar sesión como superadmin en: ' .
                       $base_url . '/admin/login.php';
        } catch (PDOException $e) {
            $error = 'Error al conectar o crear tablas: ' . $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="utf-8">
  <title>Instalación | SaaS RH Multiempresa</title>
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <style>
    body {
      font-family: system-ui, -apple-system, Segoe UI, Roboto, sans-serif;
      background:#0b1021;
      color:#e6e9f5;
      margin:0;
      padding:40px 16px;
    }
    .wrap {
      max-width:480px;
      margin:0 auto;
      background:#121734;
      padding:24px;
      border-radius:12px;
      border:1px solid #243053;
    }
    h1 { margin-top:0; font-size:22px; }
    label { display:block; margin-top:10px; font-size:14px; }
    input {
      width:100%;
      padding:10px;
      border-radius:8px;
      border:1px solid #243053;
      background:#0d1329;
      color:#e6e9f5;
      margin-top:4px;
    }
    button {
      margin-top:18px;
      width:100%;
      padding:12px;
      border-radius:10px;
      border:none;
      background:#2f6cf6;
      color:#fff;
      font-weight:700;
      cursor:pointer;
    }
    .error { background:#5c1e29; padding:10px; border-radius:8px; margin-bottom:10px; }
    .ok { background:#145c32; padding:10px; border-radius:8px; margin-bottom:10px; }
    small { color:#a9b1d6; }
  </style>
</head>
<body>
  <div class="wrap">
    <h1>Instalar SaaS RH Multiempresa</h1>
    <?php if ($error): ?>
      <div class="error"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>
    <?php if ($success): ?>
      <div class="ok"><?php echo htmlspecialchars($success); ?></div>
    <?php endif; ?>

    <form method="post">
      <label>DB Host
        <input type="text" name="db_host" value="localhost" required>
      </label>
      <label>DB Name
        <input type="text" name="db_name" required>
      </label>
      <label>DB User
        <input type="text" name="db_user" required>
      </label>
      <label>DB Password
        <input type="password" name="db_pass">
      </label>
      <label>Base URL
        <input type="text" name="base_url" placeholder="https://midominio.com/rh_saas" required>
        <small>Sin slash final. Ejemplo: https://midominio.com/rh</small>
      </label>
      <hr style="margin:20px 0;border-color:#243053;">
      <label>Nombre Superadmin
        <input type="text" name="sa_name" required>
      </label>
      <label>Email Superadmin
        <input type="email" name="sa_email" required>
      </label>
      <label>Contraseña Superadmin
        <input type="password" name="sa_pass" required>
      </label>
      <button type="submit">Instalar</button>
    </form>
  </div>
</body>
</html>
