<?php
require_once __DIR__ . '/config.php';
require_once __DIR__ . '/functions.php';
require_once __DIR__ . '/db.php';

$upload_error = '';
$upload_success = '';

// Manejo de envío de CV
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['cv_submit'])) {
    $nombre     = trim($_POST['nombre'] ?? '');
    $email      = trim($_POST['email'] ?? '');
    $telefono   = trim($_POST['telefono'] ?? '');
    $vacante_id = !empty($_POST['vacante_id']) ? (int)$_POST['vacante_id'] : null;

    if (!$nombre || !$email || empty($_FILES['cv']['name'])) {
        $upload_error = "Todos los campos obligatorios deben completarse.";
    } else {
        $allowed = ['pdf','doc','docx'];
        $ext = strtolower(pathinfo($_FILES['cv']['name'], PATHINFO_EXTENSION));

        if (!in_array($ext, $allowed)) {
            $upload_error = "Formato no permitido. Solo PDF, DOC o DOCX.";
        } else {
            $uploadDir = __DIR__ . '/uploads/cv';
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0775, true);
            }

            $filename = time() . "_" . preg_replace("/[^a-zA-Z0-9_\.]/", "", $_FILES['cv']['name']);
            $target = $uploadDir . "/" . $filename;

            if (move_uploaded_file($_FILES['cv']['tmp_name'], $target)) {
                // Guardar CV
                $stmt = $pdo->prepare("INSERT INTO cvs (nombre, email, telefono, archivo, formato) VALUES (?,?,?,?,?)");
                $stmt->execute([
                    $nombre,
                    $email,
                    $telefono,
                    'uploads/cv/' . $filename,
                    $ext
                ]);

                $cv_id = $pdo->lastInsertId();

                // Postulación
                if ($vacante_id) {
                    $stmt2 = $pdo->prepare("INSERT INTO postulaciones (cv_id, vacante_id) VALUES (?,?)");
                    $stmt2->execute([$cv_id, $vacante_id]);
                }

                $upload_success = "Tu CV se ha enviado correctamente.";
            } else {
                $upload_error = "No se pudo subir el archivo. Inténtalo de nuevo.";
            }
        }
    }
}

// Cargar vacantes abiertas
$sqlVac = "SELECT v.*, e.nombre AS empresa
           FROM vacantes v
           INNER JOIN empresas e ON e.id = v.empresa_id
           WHERE v.estado = 'abierta' AND e.is_active = 1
           ORDER BY v.created_at DESC";
$vacantes = $pdo->query($sqlVac)->fetchAll(PDO::FETCH_ASSOC);

// Empresas con al menos una vacante abierta
$sqlEmp = "SELECT e.*, 
                  COUNT(CASE WHEN v.estado = 'abierta' THEN 1 END) AS vacantes_abiertas
           FROM empresas e
           LEFT JOIN vacantes v ON v.empresa_id = e.id
           WHERE e.is_active = 1
           GROUP BY e.id
           HAVING vacantes_abiertas > 0
           ORDER BY e.nombre ASC";
$empresas = $pdo->query($sqlEmp)->fetchAll(PDO::FETCH_ASSOC);
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="utf-8">
  <title>SaaS RH Multiempresa | Vacantes & CVs</title>
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <meta name="description" content="Publica vacantes para múltiples empresas y recibe CVs en un panel centralizado.">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    body {
      background: radial-gradient(circle at top, #111827 0, #020617 55%);
      color:#e5e7eb;
    }
    .hero {
      padding:60px 16px 30px;
      text-align:center;
    }
    .hero h1 {
      font-size:clamp(1.8rem, 4vw, 2.6rem);
      margin-bottom:10px;
    }
    .hero p {
      max-width:640px;
      margin:0 auto 20px;
      color:#9ca3af;
    }
    .pill {
      display:inline-flex;
      align-items:center;
      gap:8px;
      font-size:.85rem;
      padding:6px 12px;
      border-radius:999px;
      border:1px solid #1f2937;
      color:#9ca3af;
      background:rgba(15,23,42,.75);
      margin-bottom:12px;
    }
    .btn-primary {
      background:#2563eb;
      border-color:#2563eb;
    }
    .btn-primary:hover {
      background:#1d4ed8;
      border-color:#1d4ed8;
    }
    .vacante-card {
      background:rgba(15,23,42,.9);
      border-radius:12px;
      padding:16px;
      border:1px solid #1f2937;
    }
    .vacante-card h5 { margin-bottom:6px; }
    .badge-empresa {
      background:#111827;
      border:1px solid #1f2937;
      color:#e5e7eb;
    }
    footer {
      border-top:1px solid #1f2937;
      padding:20px 0;
      color:#9ca3af;
      font-size:.9rem;
    }
    .modal-content {
      background:#020617;
      color:#e5e7eb;
      border:1px solid #1f2937;
    }
    .form-control, .form-select {
      background:#020617;
      border-color:#1f2937;
      color:#e5e7eb;
    }
  </style>
</head>
<body>

<header class="border-bottom border-secondary border-opacity-25">
  <div class="container py-3 d-flex justify-content-between align-items-center">
    <div class="d-flex align-items-center gap-2">
      <span class="badge bg-primary rounded-pill">RH</span>
      <strong>SaaS RH Multiempresa</strong>
    </div>
    <div class="d-flex gap-2">
      <a href="admin/login.php" class="btn btn-outline-light btn-sm">Entrar al panel</a>
    </div>
  </div>
</header>

<section class="hero">
  <div class="container">
    <div class="pill">
      <span>Multiempresa</span>·<span>Vacantes</span>·<span>CVs centralizados</span>
    </div>
    <h1>Centraliza el reclutamiento de varias empresas en un solo sistema</h1>
    <p>
      Publica vacantes para diferentes empresas, recibe CVs en un solo panel y
      entrega acceso solo a los administradores de cada compañía.
    </p>
    <div class="d-flex justify-content-center gap-2 flex-wrap">
      <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#cvModal">Subir mi CV</button>
      <a href="#vacantes" class="btn btn-outline-light">Ver vacantes disponibles</a>
    </div>
  </div>
</section>


<section id="empresas" class="py-4 border-bottom border-secondary-subtle">
  <div class="container">
    <div class="d-flex justify-content-between align-items-center mb-3">
      <h2 class="h5 mb-0">Empresas contratando</h2>
      <span class="text-secondary small">
        <?php echo count($empresas); ?> empresa(s) con vacantes
      </span>
    </div>

    <?php if (!$empresas): ?>
      <p class="text-secondary small">Aún no hay empresas con vacantes abiertas.</p>
    <?php else: ?>
      <div class="row row-cols-2 row-cols-md-4 g-3">
        <?php foreach ($empresas as $e): ?>
          <div class="col">
            <div class="card text-center empresa-card h-100 shadow-sm"
                 role="button"
                 data-empresa-id="<?php echo (int)$e['id']; ?>"
                 data-empresa-nombre="<?php echo h($e['nombre']); ?>">
              <?php if (!empty($e['logo'])): ?>
                <div class="p-2">
                  <img src="<?php echo h($e['logo']); ?>"
                       alt="<?php echo h($e['nombre']); ?>"
                       class="img-fluid"
                       style="max-height:60px;object-fit:contain;">
                </div>
              <?php else: ?>
                <div class="p-3 fw-semibold">
                  <?php echo h($e['nombre']); ?>
                </div>
              <?php endif; ?>
              <div class="card-body py-2">
                <span class="badge bg-primary-subtle text-primary-emphasis small">
                  <?php echo (int)$e['vacantes_abiertas']; ?> vacante(s)
                </span>
              </div>
            </div>
          </div>
        <?php endforeach; ?>
      </div>
    <?php endif; ?>
  </div>
</section>

<section id="vacantes" class="py-4">
  <div class="container">
    <div class="d-flex justify-content-between align-items-center mb-3">
      <h2 class="h4 mb-0">Vacantes abiertas</h2>
      <span class="text-secondary small"><?php echo count($vacantes); ?> vacante(s)</span>
    </div>

    <?php if (!$vacantes): ?>
      <p class="text-secondary">Por el momento no hay vacantes activas.</p>
    <?php else: ?>
      <!-- Grid cards para escritorio -->
      <div class="row row-cols-1 row-cols-md-2 row-cols-lg-3 g-3 d-none d-md-flex">
        <?php foreach ($vacantes as $v): ?>
        <div class="col">
          <div class="vacante-card h-100 d-flex flex-column">
            <div class="d-flex justify-content-between align-items-center mb-1">
              <div>
                <span class="badge badge-empresa me-1"><?php echo h($v['empresa']); ?></span>
                <span class="badge bg-secondary-subtle text-secondary-emphasis small">
                  Empresa
                </span>
              </div>
              <span class="badge bg-success-subtle text-success-emphasis">Abierta</span>
            </div>
            <h5><?php echo h($v['titulo']); ?></h5>
            <p class="mb-1 text-secondary small">
              <?php echo nl2br(h(mb_strimwidth($v['descripcion'] ?? '', 0, 140, '...'))); ?>
            </p>
            <div class="small text-secondary mb-2">
              📍 <?php echo h($v['ubicacion'] ?: 'Ubicación no especificada'); ?> ·
              🏢 <?php echo h($v['modalidad'] ?: 'Modalidad no especificada'); ?>
            </div>
            <div class="mt-auto">
              <button class="btn btn-sm btn-primary w-100"
                      data-bs-toggle="modal"
                      data-bs-target="#cvModal"
                      data-vacante="<?php echo (int)$v['id']; ?>">
                Postularme
              </button>
            </div>
          </div>
        </div>
        <?php endforeach; ?>
      </div>

      <!-- Carrusel para móviles -->
      <div id="vacantesCarousel" class="carousel slide d-md-none" data-bs-ride="carousel">
        <div class="carousel-inner">
        <?php foreach ($vacantes as $i => $v): ?>
          <div class="carousel-item <?php echo $i === 0 ? 'active' : ''; ?>">
            <div class="vacante-card mx-2">
              <div class="d-flex justify-content-between align-items-center mb-1">
                <div>
                  <span class="badge badge-empresa me-1"><?php echo h($v['empresa']); ?></span>
                  <span class="badge bg-secondary-subtle text-secondary-emphasis small">
                    Empresa
                  </span>
                </div>
                <span class="badge bg-success-subtle text-success-emphasis">Abierta</span>
              </div>
              <h5><?php echo h($v['titulo']); ?></h5>
              <p class="mb-1 text-secondary small">
                <?php echo nl2br(h(mb_strimwidth($v['descripcion'] ?? '', 0, 140, '...'))); ?>
              </p>
              <div class="small text-secondary mb-2">
                📍 <?php echo h($v['ubicacion'] ?: 'Ubicación no especificada'); ?> ·
                🏢 <?php echo h($v['modalidad'] ?: 'Modalidad no especificada'); ?>
              </div>
              <button class="btn btn-sm btn-primary w-100"
                      data-bs-toggle="modal"
                      data-bs-target="#cvModal"
                      data-vacante="<?php echo (int)$v['id']; ?>">
                Postularme
              </button>
            </div>
          </div>
        <?php endforeach; ?>
        </div>
        <button class="carousel-control-prev" type="button" data-bs-target="#vacantesCarousel" data-bs-slide="prev">
          <span class="carousel-control-prev-icon"></span>
        </button>
        <button class="carousel-control-next" type="button" data-bs-target="#vacantesCarousel" data-bs-slide="next">
          <span class="carousel-control-next-icon"></span>
        </button>
      </div>
    <?php endif; ?>
  </div>
</section>


<footer>
  <div class="container d-flex justify-content-between flex-wrap gap-2">
    <span>© <?php echo date('Y'); ?> SaaS RH Multiempresa</span>
    <span class="text-secondary">Hecho para múltiples empresas y equipos de reclutamiento.</span>
  </div>
</footer>


<!-- Modal Vacantes por Empresa -->
<div class="modal fade" id="empresaModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-lg modal-dialog-centered">
    <div class="modal-content bg-dark text-light">
      <div class="modal-header border-secondary-subtle">
        <h5 class="modal-title">
          Vacantes en <span id="empresaModalNombre"></span>
        </h5>
        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Cerrar"></button>
      </div>
      <div class="modal-body">
        <div id="empresaModalVacantes" class="vstack gap-3"></div>
        <div id="empresaModalEmpty" class="text-secondary small d-none">
          No hay vacantes abiertas actualmente en esta empresa.
        </div>
      </div>
    </div>
  </div>
</div>

<!-- Modal Subir CV -->
<div class="modal fade" id="cvModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <form class="modal-content" method="post" enctype="multipart/form-data">
      <div class="modal-header">
        <h5 class="modal-title">Postularme / Subir CV</h5>
        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Cerrar"></button>
      </div>
      <div class="modal-body">
        <?php if ($upload_error): ?>
          <div class="alert alert-danger py-2"><?php echo h($upload_error); ?></div>
        <?php elseif ($upload_success): ?>
          <div class="alert alert-success py-2"><?php echo h($upload_success); ?></div>
        <?php endif; ?>

        <input type="hidden" name="cv_submit" value="1">
        <input type="hidden" name="vacante_id" id="vacante_id_modal" value="">

        <div class="mb-2">
          <label class="form-label">Nombre completo</label>
          <input type="text" name="nombre" class="form-control" required>
        </div>
        <div class="mb-2">
          <label class="form-label">Correo</label>
          <input type="email" name="email" class="form-control" required>
        </div>
        <div class="mb-2">
          <label class="form-label">Teléfono</label>
          <input type="text" name="telefono" class="form-control">
        </div>
        <div class="mb-2">
          <label class="form-label">Vacante (opcional)</label>
          <select name="vacante_select" class="form-select" onchange="syncVacante(this.value)">
            <option value="">-- Selecciona --</option>
            <?php foreach ($vacantes as $v): ?>
              <option value="<?php echo (int)$v['id']; ?>"><?php echo h($v['titulo'] . ' · ' . $v['empresa']); ?></option>
            <?php endforeach; ?>
          </select>
        </div>

        <div class="mb-2 small">
          <div class="p-2 rounded border border-secondary-subtle bg-dark">
            <div><strong>Empresa seleccionada:</strong> <span id="cv_empresa_label">Sin empresa seleccionada</span></div>
            <div><strong>Vacante seleccionada:</strong> <span id="cv_vacante_label">Sin vacante seleccionada</span></div>
          </div>
        </div>

        <div class="mb-2">
          <label class="form-label">Archivo CV (PDF, DOC, DOCX)</label>
          <input type="file" name="cv" class="form-control" required>
        </div>
      </div>
      <div class="modal-footer">
        <button class="btn btn-primary" type="submit">Enviar CV</button>
      </div>
    </form>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
<script>
// Datos de vacantes disponibles (PHP -> JS)
const VACANTES_DATA = <?php echo json_encode($vacantes, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES); ?>;
const VACANTES_BY_ID = {};
VACANTES_DATA.forEach(v => { VACANTES_BY_ID[v.id] = v; });

function setVacanteInfo(id) {
  const empresaLabel = document.getElementById('cv_empresa_label');
  const vacanteLabel = document.getElementById('cv_vacante_label');
  const v = VACANTES_BY_ID[id];
  if (v) {
    if (empresaLabel) empresaLabel.textContent = v.empresa || 'Empresa sin nombre';
    if (vacanteLabel) vacanteLabel.textContent = v.titulo || 'Vacante sin título';
  } else {
    if (empresaLabel) empresaLabel.textContent = 'Sin empresa seleccionada';
    if (vacanteLabel) vacanteLabel.textContent = 'Sin vacante seleccionada';
  }
}

// Sincronizar cuando se muestra el modal de CV
const cvModal = document.getElementById('cvModal');
if (cvModal) {
  cvModal.addEventListener('show.bs.modal', event => {
    const button = event.relatedTarget;
    const vacanteId = button ? button.getAttribute('data-vacante') : '';
    const inputHidden = document.getElementById('vacante_id_modal');
    const selectVac = cvModal.querySelector('select[name="vacante_select"]');

    if (inputHidden) inputHidden.value = vacanteId || '';
    if (selectVac) {
      if (vacanteId) {
        selectVac.value = vacanteId;
      } else {
        selectVac.value = '';
      }
    }
    setVacanteInfo(vacanteId || null);
  });
}

function syncVacante(id) {
  const inputHidden = document.getElementById('vacante_id_modal');
  if (inputHidden) inputHidden.value = id || '';
  setVacanteInfo(id || null);
}

// Tarjetas de empresas -> abrir modal con sus vacantes
function escapeHtml(str) {
  if (!str) return '';
  return str.replace(/[&<>"']/g, function(m) {
    return ({
      '&':'&amp;',
      '<':'&lt;',
      '>':'&gt;',
      '"':'&quot;',
      "'":'&#39;'
    })[m];
  });
}

const empresaCards = document.querySelectorAll('.empresa-card');
empresaCards.forEach(card => {
  card.addEventListener('click', () => {
    const empresaId = card.getAttribute('data-empresa-id');
    const empresaNombre = card.getAttribute('data-empresa-nombre') || '';
    const modalTitleSpan = document.getElementById('empresaModalNombre');
    const listContainer = document.getElementById('empresaModalVacantes');
    const emptyMsg = document.getElementById('empresaModalEmpty');

    if (modalTitleSpan) modalTitleSpan.textContent = empresaNombre;
    if (listContainer) listContainer.innerHTML = '';

    const vacs = VACANTES_DATA.filter(v => String(v.empresa_id) === String(empresaId));

    if (!vacs.length) {
      if (emptyMsg) emptyMsg.classList.remove('d-none');
    } else {
      if (emptyMsg) emptyMsg.classList.add('d-none');
      vacs.forEach(v => {
        const item = document.createElement('div');
        item.className = 'p-3 rounded border border-secondary-subtle bg-dark-subtle';

        const desc = v.descripcion || '';
        const shortDesc = desc.length > 180 ? desc.substring(0, 180) + '…' : desc;

        item.innerHTML = `
          <div class="d-flex justify-content-between align-items-center mb-1">
            <strong>${escapeHtml(v.titulo)}</strong>
            <span class="badge bg-success-subtle text-success-emphasis">Abierta</span>
          </div>
          <div class="small text-secondary mb-1">
            📍 ${escapeHtml(v.ubicacion || 'Ubicación no especificada')} ·
            🏢 ${escapeHtml(v.modalidad || 'Modalidad no especificada')}
          </div>
          <div class="small mb-2">${escapeHtml(shortDesc)}</div>
          <button type="button" class="btn btn-sm btn-primary w-100" data-vacante="${v.id}">
            Postularme a esta vacante
          </button>
        `;

        const btn = item.querySelector('button');
        btn.addEventListener('click', () => {
          const hidden = document.getElementById('vacante_id_modal');
          const selectVac = document.querySelector('select[name="vacante_select"]');
          if (hidden) hidden.value = v.id;
          if (selectVac) selectVac.value = v.id;
          setVacanteInfo(v.id);

          const empresaModalEl = document.getElementById('empresaModal');
          if (empresaModalEl) {
            const instance = bootstrap.Modal.getInstance(empresaModalEl);
            if (instance) instance.hide();
          }
          const cvModalEl = document.getElementById('cvModal');
          if (cvModalEl) {
            const instanceCV = new bootstrap.Modal(cvModalEl);
            instanceCV.show();
          }
        });

        listContainer.appendChild(item);
      });
    }

    const empresaModalEl = document.getElementById('empresaModal');
    if (empresaModalEl) {
      const instance = new bootstrap.Modal(empresaModalEl);
      instance.show();
    }
  });
});
</script>
</body>
</html>
