<?php
require_once __DIR__ . '/header.php';
global $pdo;

// Carpeta base de imágenes (../images desde admin)
$baseDir = realpath(__DIR__ . '/../images');
$formats = [
  'square'    => 'Cuadradas (square)',
  'landscape' => 'Horizontales (landscape)',
  'portrait'  => 'Verticales (portrait)'
];

$msgSuccess = null;
$msgError   = null;

// Seguridad: helper para validar formato
function isValidFormat($fmt) {
  return in_array($fmt, ['square','landscape','portrait'], true);
}

// ==========================
//  MANEJO DE FORMULARIOS
// ==========================
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

  // SUBIR IMAGEN
  if (isset($_POST['action']) && $_POST['action'] === 'upload') {
    $format = $_POST['format'] ?? '';

    if (!isValidFormat($format)) {
      $msgError = "Formato inválido.";
    } elseif (!isset($_FILES['image']) || $_FILES['image']['error'] !== UPLOAD_ERR_OK) {
      $msgError = "Error al subir la imagen.";
    } else {
      $targetDir = $baseDir . '/' . $format;
      if (!is_dir($targetDir)) {
        @mkdir($targetDir, 0775, true);
      }

      $fileTmp  = $_FILES['image']['tmp_name'];
      $origName = $_FILES['image']['name'];

      // Extensión permitida
      $ext = strtolower(pathinfo($origName, PATHINFO_EXTENSION));
      $allowed = ['jpg','jpeg','png','webp'];

      if (!in_array($ext, $allowed, true)) {
        $msgError = "Formato no permitido. Usa jpg, jpeg, png o webp.";
      } else {
        // Nombre único pero legible
        $safeName = preg_replace('/[^a-zA-Z0-9_\-\.]/','_', pathinfo($origName, PATHINFO_FILENAME));
        $filename = $safeName . '_' . date('Ymd_His') . '.' . $ext;
        $fullPath = $targetDir . '/' . $filename;

        if (move_uploaded_file($fileTmp, $fullPath)) {
          $msgSuccess = "Imagen subida correctamente a {$format}.";
        } else {
          $msgError = "No se pudo mover el archivo subido.";
        }
      }
    }
  }

  // ELIMINAR IMAGEN
  if (isset($_POST['action']) && $_POST['action'] === 'delete') {
    $format   = $_POST['format'] ?? '';
    $filename = $_POST['filename'] ?? '';

    if (!isValidFormat($format)) {
      $msgError = "Formato inválido.";
    } elseif (!$filename) {
      $msgError = "Archivo no especificado.";
    } else {
      $targetDir = realpath($baseDir . '/' . $format);
      if ($targetDir === false) {
        $msgError = "No se encontró la carpeta del formato.";
      } else {
        $filePath = $targetDir . '/' . basename($filename);

        // Asegurar que el archivo está dentro de la carpeta
        if (strpos(realpath($filePath), $targetDir) !== 0) {
          $msgError = "Ruta inválida.";
        } elseif (is_file($filePath)) {
          if (@unlink($filePath)) {
            $msgSuccess = "Imagen eliminada correctamente.";
          } else {
            $msgError = "No se pudo eliminar el archivo.";
          }
        } else {
          $msgError = "El archivo no existe.";
        }
      }
    }
  }
}

// ==========================
//  LISTADO DE IMÁGENES
// ==========================
$imagesByFormat = [];

foreach (array_keys($formats) as $fmt) {
  $dir = $baseDir . '/' . $fmt;
  $list = [];
  if (is_dir($dir)) {
    $files = glob($dir . '/*.{jpg,jpeg,png,webp,WEBP,JPG,JPEG,PNG}', GLOB_BRACE);
    if ($files) {
      foreach ($files as $f) {
        $list[] = [
          'filename' => basename($f),
          'url'      => '../images/' . $fmt . '/' . basename($f)
        ];
      }
    }
  }
  $imagesByFormat[$fmt] = $list;
}
?>

<h1 class="h4 mb-3">Imágenes de giftcards</h1>

<?php if ($msgSuccess): ?>
  <div class="alert alert-success"><?php echo htmlspecialchars($msgSuccess); ?></div>
<?php endif; ?>
<?php if ($msgError): ?>
  <div class="alert alert-danger"><?php echo htmlspecialchars($msgError); ?></div>
<?php endif; ?>

<div class="card mb-4">
  <div class="card-body">
    <h5 class="card-title">Subir nueva imagen</h5>
    <form method="post" enctype="multipart/form-data" class="row gy-2 gx-3 align-items-end">
      <input type="hidden" name="action" value="upload">
      <div class="col-md-3">
        <label class="form-label">Formato</label>
        <select name="format" class="form-select" required>
          <option value="">Selecciona formato...</option>
          <?php foreach ($formats as $key => $label): ?>
            <option value="<?php echo htmlspecialchars($key); ?>"><?php echo htmlspecialchars($label); ?></option>
          <?php endforeach; ?>
        </select>
      </div>
      <div class="col-md-5">
        <label class="form-label">Archivo de imagen</label>
        <input type="file" name="image" class="form-control" accept=".jpg,.jpeg,.png,.webp" required>
      </div>
      <div class="col-md-2">
        <button class="btn btn-primary w-100">Subir</button>
      </div>
      <div class="col-12">
        <small class="text-muted">
          Las imágenes se guardan en <code>/images/square</code>, <code>/images/landscape</code> y <code>/images/portrait</code>.
        </small>
      </div>
    </form>
  </div>
</div>

<?php foreach ($formats as $fmtKey => $fmtLabel): ?>
  <div class="card mb-4">
    <div class="card-header d-flex justify-content-between align-items-center">
      <span><?php echo htmlspecialchars($fmtLabel); ?></span>
      <span class="badge bg-secondary">
        <?php echo count($imagesByFormat[$fmtKey]); ?> imágenes
      </span>
    </div>
    <div class="card-body">
      <?php if (empty($imagesByFormat[$fmtKey])): ?>
        <p class="text-muted mb-0">No hay imágenes en este formato.</p>
      <?php else: ?>
        <div class="row g-3">
          <?php foreach ($imagesByFormat[$fmtKey] as $img): ?>
            <div class="col-6 col-md-3 col-lg-2">
              <div class="border rounded-3 p-2 text-center">
                <img src="<?php echo htmlspecialchars($img['url']); ?>"
                     class="img-fluid rounded mb-2"
                     style="object-fit:cover; height:90px; width:100%;"
                     alt="Imagen">
                <div class="small text-truncate" title="<?php echo htmlspecialchars($img['filename']); ?>">
                  <?php echo htmlspecialchars($img['filename']); ?>
                </div>
                <form method="post" class="mt-2"
                      onsubmit="return confirm('¿Eliminar esta imagen?');">
                  <input type="hidden" name="action" value="delete">
                  <input type="hidden" name="format" value="<?php echo htmlspecialchars($fmtKey); ?>">
                  <input type="hidden" name="filename" value="<?php echo htmlspecialchars($img['filename']); ?>">
                  <button class="btn btn-sm btn-outline-danger w-100">Eliminar</button>
                </form>
              </div>
            </div>
          <?php endforeach; ?>
        </div>
      <?php endif; ?>
    </div>
  </div>
<?php endforeach; ?>

<?php require_once __DIR__ . '/footer.php'; ?>
